/***************************************************************************
 *                                                                         *
   Projekt          : Jeti 2 FrSky Telemetry Converter
 *                                                                         *
   Projekt,Version  : Version 0.1.0
 *                                                                         *
   Datum            : 11.06.2017
 *                                                                         *
   Autor            : jopl (originaly jan68)
 *                                                                         *
   Compiler-Version : Arduino 1.8.1
 *                                                                         *
   Board            : Arduino Pro Mini 16MHz 5V
 *                                                                         *
   Notes            : This sketch uses a modified HardwareSerial library
                      taken from
                      https://github.com/Bouni/Arduino/tree/
                      ide-1.5.x-hardware-serial-9-bit
                      put into
                      ..\hardware\arduino\avr\cores\arduino
 *                                                                         *
                      functions "sendRSSI" & "serializeFrsky" by T.Pfaff
 *                                                                         *
                      PIN's used:
                      11, GND to FrSky serial port
 *                                                                         *
                      Rx,Tx & GND to Jeti Transmitter Module like this:
                         Schottky    from 330R to    about
                          BAT85      470R             1k
                      Tx --|<---------[]------ Rx-----[]----- Jeti Signal
 *                                                                         *
 *                                                                         *
   ChangeLog        : 0.1.0 - first version
 *                                                                         *
 *                                                                         *
 ***************************************************************************/

//*******************
//** Header files  **
//*******************
#include <SoftwareSerial.h>

//***************
//** Constants **
//***************

//#define DEBUG01                                     // Defined in case of debug process
//#define DEBUG02                                     // Defined in case of debug process
//#define DEBUG03                                     // Defined in case of debug process
//#define DEBUG04                                     // Defined in case of debug process

#define KEYPRESSWAITINGTIME    500                // Time of button pressed

#define LED_PIN                 13
#define TXPIN                   11
#define RXPIN                   10
#define SWITCHPIN                2
#define TXPIN_DEBUG              9
#define RXPIN_DEBUG              8
#define DEBUG_PIN                7

#define AD0RANGE                55                // Range of A1-value (Receiver-Voltage) in Taranis Telemetrie (in 0,1V)  

#define PROTOCOL_HEADER       0x7E
#define PROTOCOL_PRIM         0xFD
#define PROTOCOL_TAIL         0x7E
#define PROTOCOL_RSSI         0xFE
#define PROTOCOL_STUF         0x7D
#define PROTOCOL_STUFX0       0x5E
#define PROTOCOL_STUFX1       0x5D

#define PROTOCOL_HUB_HEADER   0x5E
#define PROTOCOL_HUB_TAIL     0x5E
#define PROTOCOL_HUB_STUF     0x5D
#define PROTOCOL_HUB_STUFX0   0x3E
#define PROTOCOL_HUB_STUFX1   0x3D

#define ID_TEMPERATUR_1       0x02
#define ID_FUEL_LEVEL         0x04
#define ID_TEMPERATUR_2       0x05
#define ID_VOLTHD_BP          0x3A
#define ID_VOLTHD_AP          0x3B
#define ID_ALTITUDE_BP        0x10
#define ID_ALTITUDE_AP        0x21
#define ID_CURRENT            0x28
#define ID_CELS               0x06
#define ID_VARIO              0x30

#define ID_CAPACITY           0x29

SoftwareSerial mySerial(RXPIN, TXPIN, true);      // set up a new serial port with inverse logic for Taranis
#ifdef DEBUG01
SoftwareSerial mySerial_Debug(RXPIN_DEBUG, TXPIN_DEBUG, false); // SetUp a new serial port for debugging
#endif

//************************************************
//** Hier die benoetigten Variablen definieren  **
//************************************************

boolean LEDState = false;
byte Pointer = 0;
boolean StringOK;
boolean ConnectionOK;
byte Button = B11110000;
byte SensorType = 0;                  // SensorType: 0 no device, 1 receiver data, 2 jopl module
byte MenuPosition;                    // MenuStructure: 128 Tx, 64 Rx, 32 Mx; every submenu level => +1
byte MenuPositionOld = 128;
boolean MenuCorrect = false;
unsigned long ButtonPressTime = 0;    // For saving button pressing time

char IncomingChar;
char JetiBoxString[33];

byte ad0 = 0;
byte ad1 = 0;
byte rssi = 0;
byte jetiboxcntr = 0;

int cell01;
int cell02;
int cell03;
int alti;
int vario;

//*******************
//** SetUp routine **
//*******************
void setup() {
  Serial.begin(9600, SERIAL_9O1);         // Set up a serial line for 9-bit serial with odd parity (JetiBox)
  mySerial.begin(9600);                   // Set the data rate for the SoftwareSerial port (frSky)
#ifdef DEBUG01
  mySerial_Debug.begin(9600);             // Set the data rate for the SoftwareSerial debugging port
  pinMode(DEBUG_PIN, OUTPUT);             // Set debug pin to output
#endif
  pinMode(LED_PIN, OUTPUT);               // Set LED pin to output
  pinMode(SWITCHPIN, INPUT_PULLUP);       // Set pin to input; this pin decides, if receiver-voltage or sensor-values will be read ...
  while (Pointer < 33) {                  // Clear JetiBoxString array
    JetiBoxString[Pointer] = ' ';
    Pointer++;
  }
  delay(3000);  	                        // Wait for initialization of TX + JETI module
  digitalWrite(LED_PIN, HIGH);            // Indication of function
  Button = B11110000;                     // No button pressed
  if (!digitalRead(SWITCHPIN)) {          // Receiver info / jopl module
#ifdef DEBUG01
    mySerial_Debug.println("Receiver mode");
#endif
    while (!MenuCorrect) {                // Wait for receiver voltage screen
      JetiBoxRead();
      navigate(B01000011);
      delay(50);
    }
    SensorType = 1;                       // Set receiver info as information to read
#ifdef DEBUG01
    mySerial_Debug.println("Receiver menu achieved");
#endif
  }
  else {
#ifdef DEBUG01
    mySerial_Debug.println("Sensor mode");
#endif
    while (!SensorType) {                 // Wait for setting of sensor
      JetiBoxRead();
      navigate(B00100001);
      delay(50);
    }
#ifdef DEBUG01
    mySerial_Debug.println("Sensor mode achieved");
#endif
  }
}

//******************
//** Loop routine **
//******************
void loop() {
  JetiBoxRead();
  if (StringOK) {
    switch (SensorType) {
      case 1:
        navigate(B01000011);                   // Receiver info
        break;
      case 2:
        navigate(B00100001);                   // Data from jopl module
        break;
    }
    if (MenuCorrect) {
      LEDState = !LEDState;
      digitalWrite(LED_PIN, LEDState);
      switch (SensorType) {
        case 1:
          readReceiver();
          break;
        case 2:
          readjopl();
          break;
      }
    }
    else {
      if (!ConnectionOK) {
        readZero();
      }
    }
    jetiboxcntr++;
    if (jetiboxcntr > 2) {                    // Every n jetibox frame ---> send info
      sendFrsky();
      jetiboxcntr = 0;
    }
  }
}

//******************************
//** Functions and procedures **
//******************************

void JetiBoxRead() {
  StringOK = false;
  while (IncomingChar != -2) {                              // Start of JetiBox string
    if (Serial.available()) {
      IncomingChar = Serial.read();                           // Read the incoming chars ... wait for start byte
    }
  }
  Pointer = 0;
  while (IncomingChar != -1 && Pointer < 33) {              // Read incoming chars to string array
    if (Serial.available()) {
      IncomingChar = Serial.read();
      JetiBoxString[Pointer] = IncomingChar;
      Pointer++;
    }
  }
  if (JetiBoxString[Pointer - 1] == -1 && Pointer == 33) {  // JetiBox string array without error?
    delay(4);                                                 // Delay for button status to be sent
    Serial.write(Button);                                     // Send pressed button
    StringOK = true;                                          // Set flag
  }
#ifdef DEBUG02
  mySerial_Debug.println(JetiBoxString);
#endif
}

void navigate(byte MenuTarget) {
  MenuPositionOld = MenuPosition;                                   // Save last menu posititon
  MenuPosition = 0;                                                 // Menu is unknown at the moment
  ConnectionOK = false;                                             // No evidence of receiver or module connection
  Button = B11110000;                                               // No button is pressed
  if (JetiBoxString[3] == 'T') {                                    //    Tx ->
    MenuPosition = B10000000;                                       // Transmitter
  }
  if (JetiBoxString[3] == 'R') {                                    // <- Rx ->
    MenuPosition = B01000000;                                       // Receiver ...
    ConnectionOK = (JetiBoxString[19] == 'v');                      // ... is connected => set flag
  }
  if (JetiBoxString[0] == 'T') {                                    // Receiver type ...
    MenuPosition = B01000001;
  }
  if (JetiBoxString[0] == 'M') {                                    // Measure or setting...
    MenuPosition = B01000010;
  }
  if (JetiBoxString[0] == 'U') {                                    // Umin/Uact/Umax...
    MenuPosition = B01000011;                                       // Receiver-non-EX
  }
  if (JetiBoxString[0] == 'V') {                                    // Volt MIN/ACT/MAX.....
    MenuPosition = B01000011;                                       // Receiver-EX
  }
  if (JetiBoxString[3] == 'M') {                                    // <- Mx
    MenuPosition = B00100000;                                       // Sensor ....
    ConnectionOK = (JetiBoxString[19] == 'v');                      // ... is connected
  }
  if (JetiBoxString[1] == 'M') {                                    // Mx Connecting...
    MenuPosition = B00100001;
  }
  if (JetiBoxString[0] == 'M' && JetiBoxString[1] == '1') {         // "Alarm" at the beginning of string ... jopl module
    MenuPosition = B00100001;
    SensorType = 2;
  }
  // Test of right menu position
  ConnectionOK = (ConnectionOK || (MenuPosition & B00001111));      // Receiver or module connected (flag set before or any submenu)?
  MenuCorrect = (MenuPosition == MenuTarget);                       // Set flag of target menu position
  if (MenuPosition && MenuPositionOld != MenuPosition) {            // In case of menu change ...
    ButtonPressTime = millis() - KEYPRESSWAITINGTIME;               // ... release button (set time of button press over)
  }
  if (MenuPosition &&                                               // In case of known menu position and ...
      millis() > ButtonPressTime + KEYPRESSWAITINGTIME) {           // ... button press time over
    if (MenuPosition & MenuTarget & B11100000) {                    // In case of right device (Tx / Rx / Mx) ...
      if ((MenuPosition & B00001111) > (MenuTarget & B00001111)) {  // In case of submenu lower then target
        Button = B11010000;                                         // ... Button UP
        ButtonPressTime = millis();                                 // And set time of button press
      }
      if ((MenuPosition & B00001111) < (MenuTarget & B00001111)) {  // In case of submenu higher then target
        Button = B10110000;                                         // ... Button DOWN
        ButtonPressTime = millis();                                 // And set time of button press
      }
    }
    else {                                                          // In case of not target device
      if (MenuPosition & B00001111) {                               // In case of any submenu
        Button = B11010000;                                         // ... Button UP
        ButtonPressTime = millis() - KEYPRESSWAITINGTIME;           // And set time of button press (longer time)
      }
      else {
        if (MenuPosition > MenuTarget) {                            // In case of main menu to be left from target
          Button = B11100000;                                       // ... Button RIGHT (Tx -> Rx -> Mx)
          ButtonPressTime = millis();
        }
        else {                                                      // In case of main menu to be left from target
          Button = B01110000;                                       // ... Button LEFT (Tx <- Rx <- Mx)
          ButtonPressTime = millis();
        }
      }
    }
  }
#ifdef DEBUG03
  switch (Button) {
    case B11110000:
      mySerial_Debug.println("N");
      break;
    case B11100000:
      mySerial_Debug.println("R");
      break;
    case B11010000:
      mySerial_Debug.println("U");
      break;
    case B10110000:
      mySerial_Debug.println("D");
      break;
    case B01110000:
      mySerial_Debug.println("L");
  }
#endif
}

void readZero() {
  ad0 = 0;
  ad1 = 0;
  rssi = 0;
  cell01 = 0;
  cell02 = 0;
  cell03 = 0;
  alti = 0;
  vario = 0;
}

void readReceiver() {
  ad0 = str2valVoltage(JetiBoxString[17], JetiBoxString[19]) * 2;
  ad1 = str2valVoltage(JetiBoxString[22], JetiBoxString[24]) * 2;
  rssi = 90;
}

void readjopl() {
  int note;
  cell01 = str2valCell(JetiBoxString[16], JetiBoxString[18]);
  cell02 = str2valCell(JetiBoxString[21], JetiBoxString[23]);
  cell02 = cell02 + 0x1000;
  cell03 = str2valCell(JetiBoxString[26], JetiBoxString[28]);
  cell03 = cell03 + 0x2000;
  //alti = str2valAlti(JetiBoxString[12], JetiBoxString[13], JetiBoxString[14], JetiBoxString[15]);
  //alti = alti + 0x352B;
  vario = str2valVario(JetiBoxString[9], JetiBoxString[10], JetiBoxString[12]);
  rssi = 90;
}

byte str2valVoltage(byte X1, byte X2) {
  if ((X1 < 0x30) || (X1 > 0x39)) X1 = 0x30;
  if ((X2 < 0x30) || (X2 > 0x39)) X2 = 0x30;
  return (((X1 - 0x30) * 10) + (X2 - 0x30)) * 2;
}

int str2valCell(byte X1, byte X2) {
  if ((X1 < 0x30) || (X1 > 0x39)) X1 = 0x30;
  if ((X2 < 0x30) || (X2 > 0x39)) X2 = 0x30;
  return (((X1 - 0x30) * 10) + (X2 - 0x30)) * 50;
}

int str2valAlti(byte X1, byte X2, byte X3, byte X4) {
  if ((X1 < 0x30) || (X1 > 0x39)) X1 = 0x30;
  if ((X2 < 0x30) || (X2 > 0x39)) X2 = 0x30;
  if ((X3 < 0x30) || (X3 > 0x39)) X3 = 0x30;
  if ((X4 < 0x30) || (X4 > 0x39)) X4 = 0x30;
  return (((X1 - 0x30) * 1000) + ((X2 - 0x30) * 100) + ((X3 - 0x30) * 10) + (X4 - 0x30));
}

int str2valVario(byte X1, byte X2, byte X3) {
  if ((X2 < 0x30) || (X2 > 0x39)) X2 = 0x30;
  if ((X3 < 0x30) || (X3 > 0x39)) X3 = 0x30;
  if (X1 != 0x2D) return (((X2 - 0x30) * 100) + ((X3 - 0x30) * 10));
  else return (0 - (((X2 - 0x30) * 100) + ((X3 - 0x30) * 10)));
}

void sendFrsky() {
  sendRSSI(ad0, ad1, rssi);
  serializeFrsky(ID_CELS, cell01, false);
  serializeFrsky(ID_CELS, cell02, false);
  serializeFrsky(ID_CELS, cell03, false);
  serializeFrsky(ID_VARIO, vario, true);            // bytes in reverse order
  //serializeFrsky(ID_ALTITUDE_BP, alti, true);       // bytes in reverse order
  //serializeFrsky(ID_ALTITUDE_AP, 0, true);          // bytes in reverse order
}

static void sendRSSI(byte ad0, byte ad1, byte rssi) {
  mySerial.write(PROTOCOL_HEADER);                    // Start byte
  mySerial.write(PROTOCOL_RSSI);                      // RSSI identificator

  if (ad0 == PROTOCOL_HEADER) {                       // Byte stuffing
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX0);
  }
  else if (ad0 == PROTOCOL_STUF) {
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX1);
  }
  else {
    mySerial.write(ad0);
  }

  if (ad1 == PROTOCOL_HEADER) {
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX0);
  }
  else if (ad1 == PROTOCOL_STUF) {
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX1);
  }
  else {
    mySerial.write(ad1);
  }

  if (rssi == PROTOCOL_HEADER) {
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX0);
  }
  else if (rssi == PROTOCOL_STUF) {
    mySerial.write(PROTOCOL_STUF);
    mySerial.write(PROTOCOL_STUFX1);
  }
  else {
    mySerial.write(rssi);
  }
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write(PROTOCOL_TAIL);
}

static void serializeFrsky(byte dataID, int dataWord, boolean reverseorder) {
  byte data[2];
  byte hub[15] = {PROTOCOL_HUB_HEADER, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  byte frame[22] = {PROTOCOL_HEADER, PROTOCOL_PRIM, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  byte hub_index = 1;
  byte frame_index = 2;

  if (reverseorder) {
    data[0] = dataWord & 0x00FF;
    data[1] = (dataWord >> 8) & 0x00FF;
  }
  else {
    data[0] = (dataWord >> 8) & 0x00FF;
    data[1] = dataWord & 0x00FF;
  }


  if (dataID == PROTOCOL_HUB_HEADER) {              // Generating hub protocol with byte stuffing
    hub[hub_index++] = PROTOCOL_HUB_STUF;
    hub[hub_index++] = PROTOCOL_HUB_STUFX0;
  }
  else if (dataID == PROTOCOL_HUB_STUF) {
    hub[hub_index++] = PROTOCOL_HUB_STUF;
    hub[hub_index++] = PROTOCOL_HUB_STUFX1;
  }
  else hub[hub_index++] = dataID;

  for (byte index = 0; index < 2; index++) {
    if (data[index] == PROTOCOL_HUB_HEADER) {
      hub[hub_index++] = PROTOCOL_HUB_STUF;
      hub[hub_index++] = PROTOCOL_HUB_STUFX0;
    }
    else if (data[index] == PROTOCOL_HUB_STUF) {
      hub[hub_index++] = PROTOCOL_HUB_STUF;
      hub[hub_index++] = PROTOCOL_HUB_STUFX1;
    }
    else hub[hub_index++] = data[index];
  }

  hub[hub_index] = PROTOCOL_HUB_TAIL;

  frame[frame_index++] = hub_index + 1;
  frame[frame_index++] = 0;

  for (byte a = 0; a < (hub_index + 1); a++) {
    frame[frame_index++] = hub[a];
  }

  frame[frame_index] = PROTOCOL_TAIL;

  for (byte index = 0; index <= frame_index; index++) {
#ifdef DEBUG04
    mySerial_Debug.print(frame[index], HEX);
    mySerial_Debug.print("-");
#endif
    mySerial.write(frame[index]);
  }
#ifdef DEBUG04
  mySerial_Debug.println("OK");
#endif
}

// Example of successful data transfer
/*void sendFrskyX() {
  // RSSI
  mySerial.write(0x7E);
  mySerial.write(0xFE);
  mySerial.write(0x46); // ad0 (range is 3.3V = FF)
  mySerial.write(0x5D); // ad1
  mySerial.write(0x5C); // RQ receiver rssi (link quality - in fact RSSI, in db, hex 5C in your example means 92 decimal, max is 110)
  mySerial.write(0xBE); // MQ module rssi
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write((byte)0);
  mySerial.write(0x7E);

  // GPS Altitude
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x01);
  mySerial.write(0x22);
  mySerial.write(0x01);
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Temperature 1
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x02);
  mySerial.write(0x1A);
  mySerial.write(0x01);
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // RPM
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x03);
  mySerial.write(0x1A);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Fuel level
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x04);
  mySerial.write(0x64);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Temperature 2
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x05);
  mySerial.write(0x4A);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Cell 1 Voltage (4.20 V)
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x06);
  mySerial.write(0x08);     // High Byte (512 mV)
  mySerial.write(0x34);     // Low Byte, (2 mV)
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Cell 2 Voltage (4.10 V)
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x06);
  mySerial.write(0x18);     // High Byte (512 mV)
  mySerial.write(0x02);     // Low Byte, (2 mV)
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Cell 3 Voltage (4.00 V)
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x06);
  mySerial.write(0x27);     // High Byte (512 mV)
  mySerial.write(0xD0);     // Low Byte, (2 mV)
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Altitude After Decimal Point
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x10);
  mySerial.write(0x8D);     // Low Byte (1 m)
  mySerial.write(0x34);     // High Byte, (256 m), Zero: 34 8D 00 17
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  // Altitude After Decimal Point
  mySerial.write(0x7E);
  mySerial.write(0xFD);
  mySerial.write(0x05);
  mySerial.write((byte)0);
  mySerial.write(0x5E);
  mySerial.write(0x21);
  mySerial.write(0x17);     // Low Byte (512 mV)
  mySerial.write((byte)0);  // High Byte, (2 mV)
  mySerial.write(0x5E);
  mySerial.write(0x7E);
  }*/
