/*
 * pid.c
 *
 *  Created on: Jan 12, 2021
 *      Author: David    Original work by Jose Barros (PTDreamer), 2017
 */

#include "pid.h"
#include "tempsensors.h"
#include "settings.h"

PIDController_t pid;


void setupPID(pid_values_t* p) {
  pid.Kp =        (float)p->Kp/1000000;
  pid.Ki =        (float)p->Ki/1000000;
  pid.Kd =        (float)p->Kd/1000000;
  pid.limMinInt = (float)p->minI/100;
  pid.limMaxInt = (float)p->maxI/100;
  pid.limMin =    (float)0;                 // Min 0% (Also prevents negative PID output to the PWM)
  pid.limMax =    (float)1;                 // Max 100%
}

int32_t calculatePID(int32_t setpoint, int32_t measurement, int32_t base, bool index) {
  float dt;
  if (index) dt = (float)(HAL_GetTick() - pid.lastTime2)/1000;
  else dt = (float)(HAL_GetTick() - pid.lastTime1)/1000;
  float error = setpoint - measurement;
  pid.proportional = pid.Kp * error;                                          // Proportional term
#if defined PID_RESET_CYCLES && PID_RESET_CYCLES>0
  if(pid.reset && ++pid.reset>(PID_RESET_CYCLES-1))                               // If pid resetted, only use proportional for few cycles to avoid spikes
      pid.reset = 0;
  if(!pid.reset){                                                               // Normal PID calculation
#endif
    if (index) {
	  pid.integrator2 = pid.integrator2 + (pid.Ki*(error*dt));                      // Integral
    if (pid.integrator2 > pid.limMaxInt)                                         // Integrator clamping
      pid.integrator2 = pid.limMaxInt;
    else if (pid.integrator2 < pid.limMinInt)
      pid.integrator2 = pid.limMinInt;
    pid.derivative2 = pid.Kd*((error-pid.prevError2)/dt);
    pid.out = pid.proportional + pid.integrator2 + pid.derivative2;               // Compute output
    }
    else {
  	  pid.integrator1 = pid.integrator1 + (pid.Ki*(error*dt));                      // Integral
      if (pid.integrator1 > pid.limMaxInt)                                         // Integrator clamping
        pid.integrator1 = pid.limMaxInt;
      else if (pid.integrator1 < pid.limMinInt)
        pid.integrator1 = pid.limMinInt;
      pid.derivative1 = pid.Kd*((error-pid.prevError1)/dt);
      pid.out = pid.proportional + pid.integrator1 + pid.derivative1;               // Compute output
    }
#if defined PID_RESET_CYCLES && PID_RESET_CYCLES>0
  }
  else
    pid.out = pid.proportional;                                                 // Reset PID mode, only proportional
#endif
  if(pid.out > pid.limMax)                                                      // Apply limits
      pid.out = pid.limMax;
  else if (pid.out < pid.limMin)
      pid.out = pid.limMin;
  if (index) {
	  pid.prevMeasurement2 = measurement;
	  pid.lastTime2 = HAL_GetTick();
	  pid.prevError2 = error;
  }
  else {
	  pid.prevMeasurement1 = measurement;
	  pid.lastTime1 = HAL_GetTick();
	  pid.prevError1 = error;
  }
  return (pid.out*base);
}

void resetPID(void){
#if defined PID_RESET_CYCLES && PID_RESET_CYCLES>0
  pid.reset = 1;
#endif

  pid.prevError1 = 0;
  pid.prevError2 = 0;
  pid.prevMeasurement1 = TIP1.last_avg;
  pid.prevMeasurement2 = TIP2.last_avg;

  pid.proportional=0;
  pid.integrator1=0;
  pid.integrator2=0;
  pid.derivative1=0;
  pid.derivative2=0;
}

float getPID_D() {
  return pid.derivative1;
}
float getPID_P() {
  return pid.proportional;
}
float getPID_I() {
  return pid.integrator1;
}
float getPID_Error() {
  return pid.prevError1;
}
float getPID_Output() {
  return pid.out;
}
int32_t getPID_SetPoint() {
  return pid.lastSetpoint;
}
int32_t getPID_PresentValue() {
  return pid.lastMeasurement;
}
